using bustrax_trafilog_webAPI.Model;
using System.Text;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System.Xml;
using System.Web;

class TrafilogRequest
{
    static string apiUrl = "https://api.traffilog.mx/clients/json";
    private static IConfiguration configuration = new ConfigurationBuilder().AddJsonFile("appsettings.json", optional: false, reloadOnChange: true).Build();
    private static readonly string username = configuration["Credentials:Username"];
    private static readonly string password = configuration["Credentials:Password"];
    private static string sToken = "";
    private static string sStatus = "";
    private static string sMessage = "";
    private static bool renewToken = false;
    private static DateTime tokenExpiration;
    private static string tokenFilePath = "token.txt";

    static async Task SetToken()
    {
        if (File.Exists(tokenFilePath))
        {
            var lines = File.ReadAllLines(tokenFilePath);
            if (lines.Length == 2)
            {
                sToken = lines[0];
                tokenExpiration = DateTime.Parse(lines[1]);
                if (DateTime.Now < tokenExpiration && !renewToken)
                {
                    return;
                }
            }
        }

        string requestData = $"{{  \"action\": {{\"name\": \"user_login\",\"parameters\": {{\"login_name\": \"{username}\",\"password\": \"{password}\"}} }}}}";
        using (HttpClient httpClient = new HttpClient())
        {
            try
            {
                StringContent content = new StringContent(requestData, Encoding.UTF8, "application/json");
                HttpResponseMessage response = await httpClient.PostAsync(apiUrl, content);
                if (response.IsSuccessStatusCode)
                {
                    string responseBody = await response.Content.ReadAsStringAsync();
                    JObject jsonObject = JObject.Parse(responseBody);
                    sToken = jsonObject["response"]["properties"]["data"][0]["session_token"].ToString();
                    tokenExpiration = DateTime.Now.AddHours(24);

                    File.WriteAllText(tokenFilePath, $"{sToken}\n{tokenExpiration}");
                }
                else
                {
                    Console.WriteLine($"Error: {response.StatusCode} - {response.ReasonPhrase}");
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"Error: {ex.Message}");
            }
        }
    }

    static async Task<List<Vehicle>> GetVehicles(string _token, string _lastTime)
    {
        string requestData = $"{{\"action\": {{\"name\": \"api_get_data\",\"parameters\": [{{\"last_time\": \"{_lastTime}\",\"license_nmbr\": \"\",\"group_id\": \"\",\"version\": \"4\"}}],\"session_token\": \"{_token}\"}}}}";
        using (HttpClient httpClient = new HttpClient())
        {
            try
            {
                StringContent content = new StringContent(requestData, Encoding.UTF8, "application/json");
                HttpResponseMessage response = await httpClient.PostAsync(apiUrl, content);
                if (response.IsSuccessStatusCode)
                {
                    string responseBody = await response.Content.ReadAsStringAsync();
                    APIResponse apiResponse = JsonConvert.DeserializeObject<APIResponse>(responseBody);
                    return apiResponse.Response.Properties.Data;
                }
                else
                {
                    Console.WriteLine($"Error: {response.StatusCode} - {response.ReasonPhrase}");
                    return new List<Vehicle>();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"Error: {ex.Message}");
                return new List<Vehicle>();
            }
        }
    }

    static async Task<List<Trip>> GetTrip(string _token, string _vehicleID, string _licenseNumber, string _fromDate, string _toDate)
    {
        string requestData = $"{{\"action\": {{\"name\": \"get_vehicle_trips\",\"parameters\": [{{\"vehicle_id\": \"{_vehicleID}\",\"license_number\": \"{_licenseNumber}\",\"driver_id\": \"\",\"from_date\": \"{_fromDate}\",\"to_date\": \"{_toDate}\",\"version\": \"4\"}}],\"session_token\": \"{_token}\"}}}}";
        using (HttpClient httpClient = new HttpClient())
        {
            try
            {
                StringContent content = new StringContent(requestData, Encoding.UTF8, "application/json");
                HttpResponseMessage response = await httpClient.PostAsync(apiUrl, content);
                if (response.IsSuccessStatusCode)
                {
                    string responseBody = await response.Content.ReadAsStringAsync();
                    TripResponse apiResponse = JsonConvert.DeserializeObject<TripResponse>(responseBody);
                    return apiResponse.Response.Properties.Data;
                }
                else
                {
                    Console.WriteLine($"Error: {response.StatusCode} - {response.ReasonPhrase}");
                    sStatus = "404";
                    sMessage = response.ReasonPhrase;
                    if (response.StatusCode == System.Net.HttpStatusCode.Forbidden || response.StatusCode == System.Net.HttpStatusCode.NotFound)
                    {
                        renewToken = true;
                        await SetToken();
                        return await GetTrip(sToken, _vehicleID, _licenseNumber, _fromDate, _toDate);
                    }
                    return new List<Trip>();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"Error: {ex.Message}");
                return new List<Trip>();
            }
        }
    }

    static async Task<List<History>> GetTripHistory(string _toke, string _tripID)
    {
        string requestData = $"{{\"action\": {{\"name\": \"get_trip_locations\",\"parameters\": [{{\"drive_id\": \"{_tripID}\"}}],\"session_token\": \"{_toke}\"}}}}";
        using (HttpClient httpClient = new HttpClient())
        {
            try
            {
                StringContent content = new StringContent(requestData, Encoding.UTF8, "application/json");
                HttpResponseMessage response = await httpClient.PostAsync(apiUrl, content);
                if (response.IsSuccessStatusCode)
                {
                    string responseBody = await response.Content.ReadAsStringAsync();
                    HistoryResponse apiResponse = JsonConvert.DeserializeObject<HistoryResponse>(responseBody);
                    return apiResponse.Response.Properties.Data;
                }
                else
                {
                    Console.WriteLine($"Error: {response.StatusCode} - {response.ReasonPhrase}");
                    return new List<History>();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"Error: {ex.Message}");
                return new List<History>();
            }
        }
    }

    static async Task<List<Vehicle>> GetBulkTrips(string _toke, string _lastTime)
    {
        string requestData = $"{{\"action\": {{\"name\": \"get_vehicle_trips_extended\",\"parameters\": [{{\"last_time\": \"{_lastTime}\",\"license_nmbr\": \"\",\"group_id\": \"\",\"version\": \"4\"}}],\"session_token\": \"{_toke}\"}}}}";
        using (HttpClient httpClient = new HttpClient())
        {
            try
            {
                StringContent content = new StringContent(requestData, Encoding.UTF8, "application/json");
                HttpResponseMessage response = await httpClient.PostAsync(apiUrl, content);
                if (response.IsSuccessStatusCode)
                {
                    string responseBody = await response.Content.ReadAsStringAsync();
                    APIResponse apiResponse = JsonConvert.DeserializeObject<APIResponse>(responseBody);
                    return apiResponse.Response.Properties.Data;
                }
                else
                {
                    Console.WriteLine($"Error: {response.StatusCode} - {response.ReasonPhrase}");
                    return new List<Vehicle>();
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine($"Error: {ex.Message}");
                return new List<Vehicle>();
            }
        }
    }

    public static String GetTrips(string _vehicleID, string _licenseNumber, string _starDate, string _endDate, string _startTime, string _endTime)
    {
        SetToken().GetAwaiter().GetResult();
        List<ResultDataContainer> data = new List<ResultDataContainer>();
        ResultResponse _response = new ResultResponse();
        ResultPropertiesContainer _properties = new ResultPropertiesContainer();
        List<History> history = new List<History>();

        // Convert to UTC requested dates
        string fechaIniCompleta=_starDate+ " "+ _startTime;
        DateTime chdateRequest=Convert.ToDateTime(fechaIniCompleta);
 
        string fechaFinCompeta=_starDate+ " "+ _endTime;
        DateTime chdateRequest2=Convert.ToDateTime(fechaFinCompeta);
        
        if (chdateRequest2 < chdateRequest)
        {
            chdateRequest2= chdateRequest2.AddDays(1);

        }

        chdateRequest=chdateRequest.ToUniversalTime();
        chdateRequest2=chdateRequest2.ToUniversalTime();

        // Get Trips of the vehicle
        List<Trip> trips = GetTrip(sToken, _vehicleID, _licenseNumber, chdateRequest.ToString("yyyy-MM-dd"), chdateRequest2.ToString("yyyy-MM-dd")).GetAwaiter().GetResult();

        int totalTrips = trips.Count();
        int totalTripsIgnored = 0;
        int totalTripsPassed = 0;
        string xml = string.Empty;
        int ch_contador = 0;
        int tripsHistoryCount = 0;

        if (trips.Count() > 0)
        {
            foreach (Trip trip in trips)
            {
                ch_contador++;
                trip.Start_time = trip.Start_time != "" ? trip.Start_time : "1900-01-01T00%3A00%3A00";
                trip.End_time = trip.End_time != "" ? trip.End_time : "1900-01-01T00%3A00%3A00";
                
                string[] startTimeParts = trip.Start_time.Split('T');
                string[] endTimeParts = trip.End_time.Split('T');
                string startDayPart = startTimeParts[0];
                string endDayParts = endTimeParts[0];
                string startTimePart = startTimeParts[1].Replace("%3A", ":");
                string endTimePart = endTimeParts[1].Replace("%3A", ":");

                DateTime startRequestTime =  DateTime.Parse(startDayPart + " " + startTimePart); //pet 
                DateTime endRequestTime  = DateTime.Parse(endDayParts+ " " + endTimePart); //pet
                DateTime endTripTime = DateTime.Parse(_starDate + " " +_endTime); //paso api
                DateTime startTripTime = DateTime.Parse(_endDate + " " +_startTime); //paso api

                // NOTA: agregar validacion para cambio de dia
                if (endTripTime < startTripTime)
                {
                    endTripTime = endTripTime.AddDays(1);
                }

                if(startTripTime <= endRequestTime.AddHours(-6) && startRequestTime.AddHours(-6) <= endTripTime)
                {
                    totalTripsPassed++;
                    ResultDataContainer _data = new ResultDataContainer()
                    {
                        Vehicle_id = trip.Vehicle_id,
                        License_number = trip.License_number,
                        Driver_id = trip.Driver_id,
                        Start_time = trip.Start_time,
                        End_time = trip.End_time,
                        Start_location = trip.Start_location,
                        End_location = trip.End_location,
                        Drive_id = trip.Drive_id,
                        Distance = trip.Distance,
                        Start_latitude = trip.Start_latitude,
                        Start_longitude = trip.Start_longitude,
                        End_latitude = trip.End_latitude,
                        End_longitude = trip.End_longitude
                    };
                    _data.Trip_history = GetTripHistory(sToken, trip.Drive_id).GetAwaiter().GetResult();
                    tripsHistoryCount += _data.Trip_history.Count();
                    sStatus = "200";
                    sMessage = "OK";
                    data.Add(_data);
                }
                else
                {
                    totalTripsIgnored++;
                }
            }

            if (totalTripsIgnored == totalTrips)
            {
                sStatus = "204";
                sMessage = "Empty request";
            }

        }
        else
        {
            sStatus = sStatus != "" ? sStatus : "204";
            sMessage = sMessage != "" ? sMessage : "Empty request";
        }

        _properties.Data = data;
        _response.Response = _properties;
        xml = GenerateXmlContent(_response , sStatus, sMessage, totalTripsPassed.ToString(), tripsHistoryCount.ToString());
        return xml;
    }

    private static string GenerateXmlContent(ResultResponse resultData, string _status, string _message, string _totalTrips, string _totalHistory)
    {
        List<ResultDataContainer> _resultData = resultData.Response.Data;
        long epochTimestampDefautl = 1167609600;// 01/01/2007 12:00:00 GMT
        DateTime now = DateTime.Now;
        var stringBuilder = new StringBuilder();
        var settings = new XmlWriterSettings
        {
            //Indent = true,
            OmitXmlDeclaration = true,
            Encoding = Encoding.UTF8
        };
        using (var xmlWriter = XmlWriter.Create(stringBuilder, settings))
        {
            xmlWriter.WriteStartDocument();
            xmlWriter.WriteStartElement("root");
            xmlWriter.WriteAttributeString("REQUEST_ID", "");
            xmlWriter.WriteAttributeString("DURATION", "4");
            xmlWriter.WriteStartElement("DATASOURCE");
            xmlWriter.WriteAttributeString("DATASOURCE", "HISTORY_LOCATION");
            xmlWriter.WriteAttributeString("ACT", "REFRESH");
            xmlWriter.WriteAttributeString("PRINT_HEADER", "VEHICLE HISTORY " + now);
            xmlWriter.WriteAttributeString("PRINT_FOOTER", "");
            xmlWriter.WriteAttributeString("USER", username);
            xmlWriter.WriteAttributeString("TOKEN_EXPIRATION", tokenExpiration.ToString("yyyy/MM/dd HH:mm:ss"));
            xmlWriter.WriteAttributeString("TOKEN", sToken.ToString().Substring(sToken.Length -5));
            xmlWriter.WriteAttributeString("STATUS", _status);
            xmlWriter.WriteAttributeString("STATUS_MESSAGE", _message);
            xmlWriter.WriteAttributeString("TRIPS_COUNT", _totalTrips);
            xmlWriter.WriteAttributeString("HISTORY_COUNT", _totalHistory);
            int count = 0;
            foreach (ResultDataContainer data in _resultData)
            {
                List<History> _hist = data.Trip_history;
                foreach (History history in _hist)
                {
                    count++;
                    string decodedDatetimeStr = HttpUtility.UrlDecode(history.Time);
                    DateTime dateTime = DateTime.Parse(decodedDatetimeStr).AddHours(-6);
                    DateTime o_time = DateTime.Parse(decodedDatetimeStr);
                    string formattedDatetime = dateTime.ToString("yyyy/MM/dd HH:mm:ss");
                    string formattedo_time = o_time.ToString("yyyy/MM/dd HH:mm:ss");

                    long unixTimestamp = ((DateTimeOffset)dateTime).ToUnixTimeSeconds(); //Convert the DateTime to Unix time
                    long ORIGIN_TIME_SEC = epochTimestampDefautl - unixTimestamp;

                    xmlWriter.WriteStartElement("LOCATION");
                    xmlWriter.WriteAttributeString("OID", count.ToString());
                    xmlWriter.WriteAttributeString("LAYER_ID", "HISTORY");
                    xmlWriter.WriteAttributeString("ORIG_TIME", "" + formattedDatetime);
                    xmlWriter.WriteAttributeString("ORIG_TIME_SEC", "" + ORIGIN_TIME_SEC);
                    xmlWriter.WriteAttributeString("O_TIME", "" + formattedo_time);
                    xmlWriter.WriteAttributeString("SCHEME_DESCR", "");
                    xmlWriter.WriteAttributeString("SPEED", history.Speed);
                    xmlWriter.WriteAttributeString("LOCATION", "");
                    xmlWriter.WriteAttributeString("LATITUDE", history.Latitude);
                    xmlWriter.WriteAttributeString("LONGITUDE", history.Longitude);
                    xmlWriter.WriteAttributeString("Direction", history.Direction);
                    xmlWriter.WriteAttributeString("COLOR", "");
                    xmlWriter.WriteAttributeString("STATUS", history.Vehicle_status);
                    xmlWriter.WriteAttributeString("MILEAGE", history.Mileage);
                    xmlWriter.WriteAttributeString("DRIVER_NAME", "");
                    xmlWriter.WriteAttributeString("VEHICLE_NUMBER", history.License_number);
                    xmlWriter.WriteAttributeString("ICON_URL", "");
                    xmlWriter.WriteEndElement();
                }
            }
            xmlWriter.WriteEndElement();
            xmlWriter.WriteEndElement();
            xmlWriter.WriteEndDocument();
        }
        return stringBuilder.ToString();
    }
}